CREATE OR REPLACE PACKAGE sim2_export_project_pkg IS

  -- Author  : KHENAS
  -- Created : 2/5/01 12:30:38 PM
  -- Purpose : load project information from sim to storet
  
  /*  CHANGES:
  
   SNG 4/16/2001 - Updated to revise count of rows ready for export in IMPORT_LOG 
  
  */
  
  -- Public variable declarations
  v_export_log_seq          NUMBER(10);
  v_cur_project_is_nbr      NUMBER(8);
  v_cur_tsmgntxt_is_nbr     NUMBER(8);
  v_single_user         NUMBER(1);
  PROCEDURE export_projects (p_org_id  IN VARCHAR2, p_import_seq IN NUMBER);


END sim2_export_project_pkg;
/
CREATE OR REPLACE PACKAGE BODY sim2_export_project_pkg IS

  -- Private type declarations
  
  -- Private constant declarations

  -- Private variable declarations

  -- Function and procedure implementations
/******************************************************************************************************/
FUNCTION get_next_project_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   
   v_next_project_is_nbr NUMBER(8);
   
BEGIN
 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF  v_cur_project_is_nbr IS NULL OR v_single_user = 0 THEN
   /* get the next avail is nbr to use */
      v_next_project_is_nbr := sim2_export_pkg.get_eisn_number('TSMPROJ',p_org_id);
      v_cur_project_is_nbr := v_next_project_is_nbr ;
   ELSE
     v_next_project_is_nbr := v_cur_project_is_nbr  + 1;
     v_cur_project_is_nbr := v_next_project_is_nbr;
   END IF;
   
   RETURN v_next_project_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next project is number.');   
     RETURN NULL;
   
END get_next_project_is_number;

/******************************************************************************************************/
FUNCTION get_next_tsmgntxt_is_number (p_org_id IN VARCHAR2) RETURN NUMBER IS
   

   v_next_tsmgntxt_is_nbr NUMBER(8);
   
BEGIN

 v_single_user := SIM2_EXPORT_PKG.check_single_user; -- Returns a 1 if it is a single user environment

   IF  v_cur_tsmgntxt_is_nbr IS NULL OR v_single_user = 0 THEN
   /* get the next avail is nbr to use */
      v_next_tsmgntxt_is_nbr := sim2_export_pkg.get_eisn_number('TSMGNTXT',p_org_id);
      v_cur_tsmgntxt_is_nbr := v_next_tsmgntxt_is_nbr ;
   ELSE
     v_next_tsmgntxt_is_nbr := v_cur_tsmgntxt_is_nbr  + 1;
     v_cur_tsmgntxt_is_nbr := v_next_tsmgntxt_is_nbr;
   END IF;
   
   RETURN v_next_tsmgntxt_is_nbr;

EXCEPTION
   WHEN OTHERS THEN
     sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - in get next tsmgntxt is number.');   
     RETURN NULL;
   
END get_next_tsmgntxt_is_number;

/******************************************************************************************************/  
FUNCTION create_tsmgntxt_rec (p_tsmgntxt_rec IN TSMGNTXT%ROWTYPE) RETURN NUMBER IS
    v_is_number NUMBER(8);
BEGIN
      
    v_is_number := get_next_tsmgntxt_is_number(p_tsmgntxt_rec.TSMGNTXT_ORG_ID);
  INSERT INTO TSMGNTXT (
    TSMGNTXT_IS_NUMBER,
    TSMGNTXT_ORG_ID,
    TABLE_NAME,
    DESCRIPTION_NAME,
    DESCRIPTION_TEXT,
    D_USERID_CODE,
    D_LAST_UPDATE_TS,
    TSRRSULT_IS_NUMBER,
    TSRRSULT_ORG_ID,
    TSMPROJ_IS_NUMBER,
    TSMPROJ_ORG_ID,
    TSMPROGM_IS_NUMBER,
    TSMPROGM_ORG_ID,
    TSRGRCFG_IS_NUMBER,
    TSRGRCFG_ORG_ID
    )
  VALUES(
    v_is_number,       /* TSMGNTXT_IS_NUMBER */
    p_tsmgntxt_rec.TSMGNTXT_ORG_ID,          /* TSMGNTXT_ORG_ID */
    p_tsmgntxt_rec.TABLE_NAME,               /* TABLE_NAME */
    p_tsmgntxt_rec.DESCRIPTION_NAME,         /* DESCRIPTION_NAME */
    p_tsmgntxt_rec.DESCRIPTION_TEXT,         /* DESCRIPTION_TEXT */
    p_tsmgntxt_rec.D_USERID_CODE,            /* D_USERID_CODE */
    p_tsmgntxt_rec.D_LAST_UPDATE_TS,         /* D_LAST_UPDATE_TS */
    p_tsmgntxt_rec.TSRRSULT_IS_NUMBER,       /* TSRRSULT_IS_NUMBER */
    p_tsmgntxt_rec.TSRRSULT_ORG_ID,          /* TSRRSULT_ORG_ID */
    p_tsmgntxt_rec.TSMPROJ_IS_NUMBER,        /* TSMPROJ_IS_NUMBER */
    p_tsmgntxt_rec.TSMPROJ_ORG_ID,           /* TSMPROJ_ORG_ID */
    p_tsmgntxt_rec.TSMPROGM_IS_NUMBER,       /* TSMPROGM_IS_NUMBER */
    p_tsmgntxt_rec.TSMPROGM_ORG_ID,          /* TSMPROGM_ORG_ID */
    p_tsmgntxt_rec.TSRGRCFG_IS_NUMBER,       /* TSRGRCFG_IS_NUMBER */
    p_tsmgntxt_rec.TSRGRCFG_ORG_ID           /* TSRGRCFG_ORG_ID */
    );
    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
      sim2_export_pkg.UPDATE_TSMEISN(p_tsmgntxt_rec.TSMGNTXT_ORG_ID,'TSMGNTXT', v_cur_tsmgntxt_is_nbr);
      COMMIT;
    END IF;
  RETURN 1;
  
EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in create tsmgntxt rec.');    
    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
       COMMIT;
    END IF;
    RETURN 0;

END create_tsmgntxt_rec;

/******************************************************************************************************/  
FUNCTION create_project (p_project_rec IN TSMPROJ%ROWTYPE) RETURN NUMBER IS

BEGIN
      
  INSERT INTO TSMPROJ(
    TSMPROJ_IS_NUMBER,
    TSMPROJ_ORG_ID,
    IDENTIFICATION_CD,
    NAME,
    START_DATE,
    PLANNED_DURATION,
    D_USERID_CODE,
    D_LAST_UPDT_TS,
    TSMORGAN_IS_NUMBER  
    )
  VALUES (
    p_project_rec.TSMPROJ_IS_NUMBER,      /* TSMPROJ_IS_NUMBER */
    p_project_rec.TSMPROJ_ORG_ID,         /* TSMPROJ_ORG_ID */
    p_project_rec.IDENTIFICATION_CD,      /* IDENTIFICATION_CD */
    p_project_rec.NAME,                   /* NAME */
    p_project_rec.START_DATE,             /* START_DATE */
    p_project_rec.PLANNED_DURATION,       /* PLANNED_DURATION */
    p_project_rec.D_USERID_CODE,          /* D_USERID_CODE */
    p_project_rec.D_LAST_UPDT_TS,         /* D_LAST_UPDT_TS */
    p_project_rec.TSMORGAN_IS_NUMBER      /* TSMORGAN_IS_NUMBER */  
    );

    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
       sim2_export_pkg.UPDATE_TSMEISN(p_project_rec.TSMPROJ_ORG_ID,'TSMPROJ', v_cur_project_is_nbr);
       COMMIT;
    END IF;
    
  RETURN 1;

EXCEPTION
  WHEN OTHERS THEN
    sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - in create project.');
    IF v_single_user = 0 THEN -- If it is a multi-user environment, commit after each insert
       COMMIT;
    END IF;
    RETURN 0;
  
END create_project;

/******************************************************************************************************/  
PROCEDURE export_projects (p_org_id  IN VARCHAR2, p_import_seq IN NUMBER) IS

  v_success                      NUMBER(1);
  
  v_project_cnt                  NUMBER(10) := 0;
  v_project_is_nbr               NUMBER(8);
  v_tsmgntxt_is_nbr              NUMBER(8);
  v_proj_rec                     TSMPROJ%ROWTYPE;
  v_proj_copy_rec                TSMPROJ%ROWTYPE;  
  v_tsmgntxt_rec                 TSMGNTXT%ROWTYPE;
  v_tsmgntxt_copy_rec            TSMGNTXT%ROWTYPE;
  
  CURSOR c_get_projects(p_import_seq IN NUMBER) IS
    SELECT SPJ_SEQ, SPJ_ID, SPJ_NAME, SPJ_START_DATE, SPJ_DURATION, SPJ_PURPOSE, SPJ_CONTACT,
           SPJ_ORG_ID, SPJ_TSMORGAN_IS_NUMBER
      FROM SIM_PROJECTS
      WHERE SPJ_IMPORT_SEQ = p_import_seq AND SPJ_STATUS = 'A';

BEGIN

  BEGIN
    /* set up export log */
    v_export_log_seq := sim2_export_pkg.GET_EXPORT_LOG_SEQ;
 
    INSERT INTO sim_export_log(
      SEL_SEQ,
      SEL_EXPORTED_BY,
      SEL_EXPORTED_ON,
      SEL_FILENAME,
      SEL_ORG_ID,
      SEL_EXPORT_TYPE
      )
    VALUES(
      v_export_log_seq,
      USER,
      SYSDATE,
      'Direct load: '||p_import_seq,
      p_org_id,
      'P'
      );

    COMMIT;
      
  EXCEPTION
    WHEN OTHERS THEN
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' - setting up export log record.');
  END;

DBMS_OUTPUT.PUT_LINE('After set up of export log');       

  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Start: '||to_char(sysdate,'HH24:MI:SS'));

  /* loop through projects */
  FOR v_project IN c_get_projects(p_import_seq) LOOP

    /* create project record */
    BEGIN
      v_project_is_nbr := get_next_project_is_number(p_org_id);
      
      v_proj_rec.TSMPROJ_IS_NUMBER := v_project_is_nbr;
      v_proj_rec.TSMPROJ_ORG_ID := p_org_id;
      v_proj_rec.IDENTIFICATION_CD := v_project.SPJ_ID;
      v_proj_rec.NAME := v_project.SPJ_NAME;
      v_proj_rec.START_DATE := v_project.SPJ_START_DATE;
      v_proj_rec.PLANNED_DURATION := v_project.SPJ_DURATION;
      v_proj_rec.D_USERID_CODE := USER;
      v_proj_rec.D_LAST_UPDT_TS := SYSDATE;
      v_proj_rec.TSMORGAN_IS_NUMBER := v_project.SPJ_TSMORGAN_IS_NUMBER;    
      
       v_success := create_project(v_proj_rec);
      
      v_project_cnt := v_project_cnt + 1;
      
    EXCEPTION
      WHEN OTHERS THEN
        sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - creating project record.');
    END;
    
    /* create tsmgntxt record - project purpose */
    IF v_project.SPJ_PURPOSE IS NOT NULL THEN
      BEGIN
      
        v_tsmgntxt_rec.TSMGNTXT_ORG_ID := p_org_id;
        v_tsmgntxt_rec.TABLE_NAME := 'TSMPROJ';
        v_tsmgntxt_rec.DESCRIPTION_NAME := 'PURPOSE';
        v_tsmgntxt_rec.DESCRIPTION_TEXT := v_project.SPJ_PURPOSE;
        v_tsmgntxt_rec.D_USERID_CODE := USER;
        v_tsmgntxt_rec.D_LAST_UPDATE_TS := SYSDATE;
        v_tsmgntxt_rec.TSRRSULT_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSRRSULT_ORG_ID := NULL;
        v_tsmgntxt_rec.TSMPROJ_IS_NUMBER := v_project_is_nbr;
        v_tsmgntxt_rec.TSMPROJ_ORG_ID := p_org_id;
        v_tsmgntxt_rec.TSMPROGM_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSMPROGM_ORG_ID := NULL;
        v_tsmgntxt_rec.TSRGRCFG_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSRGRCFG_ORG_ID := NULL;
             
        v_success := create_tsmgntxt_rec(v_tsmgntxt_rec);
      
      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - creating project purpose record.');
      END;
    END IF;
    
    /* create tsmgntxt record - project contact */
    IF v_project.SPJ_CONTACT IS NOT NULL THEN
      BEGIN
        v_tsmgntxt_rec := v_tsmgntxt_copy_rec;      
      
    
        v_tsmgntxt_rec.TSMGNTXT_ORG_ID := p_org_id;
        v_tsmgntxt_rec.TABLE_NAME := 'TSMPROJ';
        v_tsmgntxt_rec.DESCRIPTION_NAME := 'WORKPLAN';
        v_tsmgntxt_rec.DESCRIPTION_TEXT := v_project.SPJ_CONTACT;
        v_tsmgntxt_rec.D_USERID_CODE := USER;
        v_tsmgntxt_rec.D_LAST_UPDATE_TS := SYSDATE;
        v_tsmgntxt_rec.TSRRSULT_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSRRSULT_ORG_ID := NULL;
        v_tsmgntxt_rec.TSMPROJ_IS_NUMBER := v_project_is_nbr;
        v_tsmgntxt_rec.TSMPROJ_ORG_ID := p_org_id;
        v_tsmgntxt_rec.TSMPROGM_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSMPROGM_ORG_ID := NULL;
        v_tsmgntxt_rec.TSRGRCFG_IS_NUMBER := NULL;
        v_tsmgntxt_rec.TSRGRCFG_ORG_ID := NULL;
            
        v_success := create_tsmgntxt_rec(v_tsmgntxt_rec);
      
      EXCEPTION
        WHEN OTHERS THEN
          sim2_export_pkg.write_error(v_export_log_seq, SQLERRM||' - creating project contact record.');
      END;
    END IF;    
    
    /* update project table with project is number and org id, export info*/
    UPDATE SIM_PROJECTS SET
      SPJ_TSMPROJ_IS_NUMBER = v_project_is_nbr,
      SPJ_TSMPROJ_ORG_ID = RPAD(p_org_id,8),
      SPJ_EXPORT_SEQ = v_export_log_seq,
      SPJ_STATUS = 'E'
      WHERE SPJ_SEQ = v_project.SPJ_SEQ;

    /* wipe out records holding project info so they are clean for the next one */  
    v_proj_rec := v_proj_copy_rec;
    v_tsmgntxt_rec := v_tsmgntxt_copy_rec;
        
  END LOOP;
  
  /* update export log with count */
  UPDATE SIM_EXPORT_LOG SET SEL_ACTIVITY_CNT = v_project_cnt 
  WHERE SEL_SEQ = v_export_log_seq;
  
  /* update the import log with export info 
     SNG 4/16/01 - added DECREMENT of sil_rows_ready */
  UPDATE SIM_IMPORT_LOG SET 
     SIL_EXPORTED_ON = SYSDATE,
     SIL_EXPORTED_BY = USER,
     SIL_ROWS_READY = SIL_ROWS_READY - v_project_cnt
  WHERE SIL_SEQ = P_IMPORT_SEQ;
  
  /*  SNG 4/16/01 - Combined in above update statement
  UPDATE SIM_IMPORT_LOG SET SIL_EXPORTED_BY = USER
  WHERE SIL_SEQ = P_IMPORT_SEQ;
  */
  
  /* update counters */
  /* increment the is number in the TSMEISN table to the next one avail for manual entry */
  sim2_export_pkg.UPDATE_TSMEISN(p_org_id,'TSMGNTXT', v_cur_tsmgntxt_is_nbr);
  sim2_export_pkg.UPDATE_TSMEISN(p_org_id,'TSMPROJ', v_cur_project_is_nbr);
  
  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Exported '||v_project_cnt||' projects.');
  sim2_export_pkg.WRITE_ERROR(v_export_log_seq, 'Finish: '||to_char(sysdate,'HH24:MI:SS'));

  COMMIT;

EXCEPTION
   WHEN OTHERS THEN
      sim2_export_pkg.WRITE_ERROR(v_export_log_seq, SQLERRM||' -  Error in sim2_export_project_pkg.export_projects');


END export_projects;

/******************************************************************************************************/
END sim2_export_project_pkg;
/
